#ifndef _VECTOR_H_
#define _VECTOR_H_

#include <math.h>

#include "matrix.h"

#ifndef M_PI
static const double M_PI = 3.1415926535897932384626433832795;
#endif

namespace bil
{
class vector
{
public:
	float x, y, z;
	
	vector() { }
	
	vector(const float x, const float y, const float z)
	{
		this->x = x;
		this->y = y;
		this->z = z;
	}

	~vector() { }

	vector operator+(const vector v)
	{
		return vector(x + v.x, y + v.y, z + v.z);
	}

	vector& operator+=(const vector v)
	{
		x += v.x;
		y += v.y;
		z += v.z;

		return *this;
	}
	
	vector operator-()
	{
		return vector(-x, -y, -z);
	}

	vector operator-(const vector v)
	{
		return vector(x - v.x, y - v.y, z - v.z);
	}

	vector& operator-=(const vector v)
	{
		x -= v.x;
		y -= v.y;
		z -= v.z;

		return *this;
	}
	
	float operator*(const vector v)
	{
		return x * v.x + y * v.y + z * v.z;
	}


	vector operator*(matrix m)
	{
		return vector(x * m(0, 0) + y * m(0, 1) + z * m(0, 2) + m(0, 3),
		              x * m(1, 0) + y * m(1, 1) + z * m(1, 2) + m(1, 3),
		              x * m(2, 0) + y * m(2, 1) + z * m(2, 2) + m(2, 3));
	}

	inline vector& operator*=(matrix m)
	{
		float nx, ny, nz;

		nx = x * m(0, 0) + y * m(0, 1) + z * m(0, 2) + m(0, 3);
		ny = x * m(1, 0) + y * m(1, 1) + z * m(1, 2) + m(1, 3);
		nz = x * m(2, 0) + y * m(2, 1) + z * m(2, 2) + m(2, 3);

		x = nx;
		y = ny;
		z = nz;

		return *this;
	}
	
	vector operator*(float f)
	{
		return vector(x * f, y * f, z * f);
	}

	vector& operator*=(float f)
	{
		x *= f;
		y *= f;
		z *= f;

		return *this;
	}

	vector operator/(float f)
	{
		return vector(x / f, y / f, z / f);
	}

	vector& operator/=(float f)
	{
		x /= f;
		y /= f;
		z /= f;

		return *this;
	}
	
	vector& operator=(const vector v)
	{
		x = v.x;
		y = v.y;
		z = v.z;

		return *this;
	}
	
	bool operator==(const vector v)
	{
		if((v.x < x - 1e-5 || v.x > x + 1e-5)
		|| (v.y < y - 1e-5 || v.y > y + 1e-5)
		|| (v.z < z - 1e-5 || v.z > z + 1e-5))
			return false;
		
		return true;
	}

	bool operator!=(const vector v)
	{
		if((v.x < x - 1e-5 || v.x > x + 1e-5)
		|| (v.y < y - 1e-5 || v.y > y + 1e-5)
		|| (v.z < z - 1e-5 || v.z > z + 1e-5))
			return true;
		
		return false;
	}

	vector cross(const vector v)
	{
		return vector(y * v.z - v.y * z,
		              z * v.x - v.z * x,
					  x * v.y - v.x * y);
	}

	float magnitude()
	{
		return float(sqrt(x * x + y * y + z * z));
	}

	vector& normalize()
	{
		float m = magnitude();

		x /= m;
		y /= m;
		z /= m;
		
		return *this;
	}
};
};

#endif

