/*************************************************************************************************
*
*	Title:	WDDraw.cpp
*	Desc:	Provides Windowed DirectDraw Virtual Video driver
*	
*	Note:	
*			
**************************************************************************************************/

#include <Windows.h>
#include <memory.h>
#include <stdio.h>
#include <ddraw.h>
#include "Winmain.h"
#include "VVideo.h"

/*************************************************************************************************/
// Defines
/*************************************************************************************************/

/*************************************************************************************************/
// External Data
/*************************************************************************************************/

/*************************************************************************************************/
// Global Data
/*************************************************************************************************/

/*************************************************************************************************/
// Module Data
/*************************************************************************************************/

static Color32_t	*m_VPage = NULL;

// DDraw surfaces
static LPDIRECTDRAW m_DDraw = NULL;
static LPDIRECTDRAWSURFACE m_FrontBuffer = NULL;
static LPDIRECTDRAWSURFACE m_BackBuffer = NULL;
static LPDIRECTDRAWCLIPPER m_Clipper = NULL;

// color space masks
static unsigned int	m_RedMask, m_RedRShift, m_RedLShift;
static unsigned int	m_GreenMask, m_GreenRShift, m_GreenLShift;
static unsigned int	m_BlueMask, m_BlueRShift, m_BlueLShift;
static int m_DesktopBpp;

/*************************************************************************************************/
// Functions
/*************************************************************************************************/

static int GetLowestBit( unsigned int Value );
static int GetHighestBit( unsigned int Value );

/*************************************************************************************************
*
*	Function:	WinDDraw_Open()
*
*	Desc:		Creates our Surface
*
*	Notes:		
*
***************************************************************************************************/
int WinDDraw_Open( void )
{
	HRESULT hr;
	int Width, Height, Bpp;
	DDSURFACEDESC ddsd;
	DDPIXELFORMAT PixelFormat;

	// init vars
	Width = 320;
	Height = 240;
	Bpp = 16;

	// create DDraw object
	hr = DirectDrawCreate( NULL, &m_DDraw, NULL );
	if (FAILED(hr)) return 1;

	// set cooperative level
	hr = m_DDraw->SetCooperativeLevel( g_hWnd, DDSCL_NORMAL );
	if (FAILED(hr)) goto error;

	// create front and back buffers
	memset( &ddsd, 0, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS;
	ddsd.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY | DDSCAPS_PRIMARYSURFACE;
	hr = m_DDraw->CreateSurface( &ddsd, &m_FrontBuffer, NULL );
	if (FAILED(hr)) goto error;
	
	// create back buffer
	memset( &ddsd, 0, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	ddsd.dwFlags = DDSD_CAPS | DDSD_WIDTH | DDSD_HEIGHT;
	ddsd.dwWidth = Width;
	ddsd.dwHeight = Height;
	ddsd.ddsCaps.dwCaps = DDSCAPS_VIDEOMEMORY | DDSCAPS_OFFSCREENPLAIN;
	hr = m_DDraw->CreateSurface( &ddsd, &m_BackBuffer, NULL );
	if (FAILED(hr)) goto error;

	// get info on the front buffer pixel format
	memset( &PixelFormat, 0, sizeof(PixelFormat) );
	PixelFormat.dwSize = sizeof(PixelFormat);
	hr = m_FrontBuffer->GetPixelFormat( &PixelFormat );
	if (FAILED(hr)) goto error;

	// save the bpp of desktop
	m_DesktopBpp = PixelFormat.dwRGBBitCount;

	// setup color space shifts/masks
	m_RedMask = PixelFormat.dwRBitMask;
	m_RedRShift = 8 - (GetHighestBit( PixelFormat.dwRBitMask ) - GetLowestBit( PixelFormat.dwRBitMask ));
	m_RedLShift = GetLowestBit( PixelFormat.dwRBitMask );

	m_GreenMask = PixelFormat.dwGBitMask;
	m_GreenRShift = 8 - (GetHighestBit( PixelFormat.dwGBitMask ) - GetLowestBit( PixelFormat.dwGBitMask ));
	m_GreenLShift = GetLowestBit( PixelFormat.dwGBitMask );

	m_BlueMask = PixelFormat.dwBBitMask;
	m_BlueRShift = 8 - (GetHighestBit( PixelFormat.dwBBitMask ) - GetLowestBit( PixelFormat.dwBBitMask ));
	m_BlueLShift = GetLowestBit( PixelFormat.dwBBitMask );

	// create clipper
	hr = m_DDraw->CreateClipper( 0, &m_Clipper, NULL);
	if (FAILED(hr)) goto error;

	// attach clipper to window
	hr = m_Clipper->SetHWnd( 0, g_hWnd );
	if (FAILED(hr)) goto error;

	// attach clipper to front buffer
	hr = m_FrontBuffer->SetClipper( m_Clipper );
	if (FAILED(hr)) goto error;

	// allocate vpage
	m_VPage = (Color32_t *)malloc( 320*240*sizeof(Color32_t) );
	if (!m_VPage) goto error;

	return 0;

error:
	OutputDebugString( "DDraw Error" );

	if (m_Clipper)
	{
		m_Clipper->Release();
		m_Clipper = NULL;
	}

	if (m_BackBuffer)
	{
		m_BackBuffer->Release();
		m_BackBuffer = NULL;
	}

	if (m_FrontBuffer)
	{
		m_FrontBuffer->Release();
		m_FrontBuffer = NULL;
	}

	if (m_DDraw)
	{
		m_DDraw->SetCooperativeLevel( g_hWnd, DDSCL_NORMAL );
		m_DDraw->RestoreDisplayMode();
		m_DDraw->Release();
		m_DDraw = NULL;
	}

	if (m_VPage)
	{
		free( m_VPage );
		m_VPage = NULL;
	}
	
	return 1;
}

/*************************************************************************************************
*
*	Function:	WinDDraw_Close()
*
*	Desc:		Releases all resources
*
*	Notes:		
*
***************************************************************************************************/
void WinDDraw_Close( void )
{
	// free clipper
	if (m_Clipper)
	{
		m_Clipper->Release();
		m_Clipper = NULL;
	}

	// release back buffer
	if (m_BackBuffer)
	{
		m_BackBuffer->Release();
		m_BackBuffer = NULL;
	}

	// relese front buffer
	if (m_FrontBuffer)
	{
		m_FrontBuffer->Release();
		m_FrontBuffer = NULL;
	}

	// release DDraw
	if (m_DDraw)
	{
		// reset coop
		m_DDraw->Release();
		m_DDraw = NULL;
	}

	// release virtual page
	if (m_VPage)
	{
		free( m_VPage );
		m_VPage = NULL;
	}
}

/*************************************************************************************************
*
*	Function:	WinDDraw_Flip()
*
*	Desc:		Shows the VPage to the world
*
*	Notes:		
*
***************************************************************************************************/
void WinDDraw_Flip( void )
{
	HRESULT hr;
	DDSURFACEDESC ddsd;
	RECT rect;
	unsigned char *Scanline;
	unsigned char *Dest;
	Color32_t *Src;
	int x, y;
	unsigned r, g, b;

	// if were not the active app
	if (g_hWnd != GetForegroundWindow() ) return;

	// lock the surface
	memset( &ddsd, 0, sizeof(ddsd) );
	ddsd.dwSize = sizeof(ddsd);
	hr = m_BackBuffer->Lock( NULL, &ddsd, DDLOCK_SURFACEMEMORYPTR | DDLOCK_WAIT, NULL);
	if (!FAILED(hr))
	{
		// setup pointers
		Scanline = (unsigned char *)ddsd.lpSurface;
		Src = m_VPage;

		// for all scanlines
		for (y=0; y < 240; y++)
		{
			Dest = Scanline;

			// for all pixels in the scan
			for (x=0; x < 320; x++)
			{
				// conver 888 to whatever format the display is in
				r = (((Src->r >> m_RedRShift) << m_RedLShift) & m_RedMask);
				g = (((Src->g >> m_GreenRShift) << m_GreenLShift) & m_GreenMask);
				b = (((Src->b >> m_BlueRShift) << m_BlueLShift) & m_BlueMask);
				Src++;

				// write to back buffer
				
				// ugly data cast
				if (m_DesktopBpp == 16)
				{
					*((unsigned short *)Dest) = r+g+b;
					Dest += 2;
				}
				else if (m_DesktopBpp == 24)
				{
					*((unsigned int *)Dest) = r+g+b;
					Dest+=3;
				}
				else if (m_DesktopBpp == 32)
				{
					*((unsigned int *)Dest) = r+g+b;
					Dest += 4;
				}
			}
			// NOTE: lPith might NOT be equal to 320*2.
			Scanline += ddsd.lPitch;
		}
		hr = m_BackBuffer->Unlock(NULL);
		if (FAILED(hr)) OutputDebugString("BackBuffer->Unlock() Failed!");
	}

	// Get the the window dimentions
	GetWindowRect(g_hWnd, &rect);

	// specify
	hr = m_FrontBuffer->Blt( &rect, m_BackBuffer, NULL, DDBLT_WAIT, NULL);
	if (FAILED(hr)) OutputDebugString( "Frontbuffer->Blit FAILED!\n" );
}

/*************************************************************************************************
*
*	Function:	WinDDraw_GetAddress()
*
*	Desc:		Gets the address of our Virtual page
*
*	Notes:		
*
***************************************************************************************************/
Color32_t *WinDDraw_GetAddress( void )
{
	return m_VPage;
}

/*************************************************************************************************
*
*	Function:	GetLowestBit()
*
*	Desc:		Gets the lowest significant bit position
*
*	Notes:		
*
***************************************************************************************************/
static int GetLowestBit( unsigned int Value )
{
	int Position;

	// sanity check
	if (Value == 0) return 0;

	Position = 0;
	while ( (Value&1) == 0)
	{
		Position++;
		Value = Value>> 1;
	}

	return Position;
}

/*************************************************************************************************
*
*	Function:	GetHighestBit()
*
*	Desc:		Gets the highest significant bit position
*
*	Notes:		
*
***************************************************************************************************/
static int GetHighestBit( unsigned int Value )
{
	int Position;

	// sanity check
	if (Value == 0) return 0;

	Position = 0;
	while ( (Value&0x80000000) == 0)
	{
		Position++;
		Value = Value<< 1;
	}

	return 32 - Position;
}
